-- The function 'CalculateFluorescenceSEmatrix()' determines the fluorescence self-energy matrix for a
-- given set of states.

-- Input: - energy interval ([wMin, wMax]) and step width (dw)
--        - number of states for which the self-energy matrix is calculated (NStates)
--        - list of energies of electronic eigenstates after fluorescence decay (int_eigenvals)
--        - list of electric multipole transition operators given in terms of matrices (opmat_list)
--        - list of interpolated radial integrals (radint_list) which belong to the corresponding operators in opmat_list

-- Output: Imaginary part of fluorescence self-energy matrix (ImSigma_w)

function CalculateFluorescenceSEmatrix(wMin,wMax,dw,NStates,int_eigenvals,opmat_list,radint_list)

	ImSigma_w = {}

	for w = wMin, wMax, dw do	
		
		ImSigma = {}
	
		n_i = NStates+1
		n_f = NStates+(#int_eigenvals)
		
		-- loop over all (electron_ic) states after fluorescence decay 
		for n = n_i, n_f do
						
			E_n = int_eigenvals[n-NStates]
			w_p = w - E_n
				
			-- state contributes only if photon energy is greater than zero
			if w_p > 0 then

				-- in_itialize OpMatE (E1 transition operator) 
				OpMatE =  opmat_list[1][1] * radint_list[1][1](w_p) + opmat_list[1][2] * radint_list[1][2](w_p) 
				-- ...and add up all contributions
				if #opmat_list > 1 then
					for r = 2, #opmat_list do
						OpMatE = OpMatE + opmat_list[r][1] * radint_list[r][1](w_p)  + opmat_list[r][2] * radint_list[r][2](w_p)
					end
				end
			
				-- express transition operator as matrix
				OpMatE = Matrix.ToTable(OpMatE)
			
				for i = 1, NStates do

					if n == n_i then
						ImSigma[i] = {}
					end

					for j = 1, NStates do

						if n == n_i then
							ImSigma[i][j] = 0
						end
												
						ImSigma[i][j] = ImSigma[i][j] - 4/137 * Pi * w_p * OpMatE[n][j] * OpMatE[n][i]

					end 
				
				end	
	
			end	
		
		end
					
		-- save the result for each w
		ImSigma_w[#ImSigma_w+1] = ImSigma
	
	end

	return ImSigma_w

end


